/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Implemnation of the "Device/File Explorer" dialog

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2006-04-23  initial version

**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file FileExplorerDlg.cpp
///  Implemnation of the "Device/File Explorer" dialog
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "cifXTest.h"
#include "FileExplorerDlg.h"
#include "cifXTestDlg.h"
#include "rcX_Public.h"
#include ".\fileexplorerdlg.h"

#define COLUMN_FILESIZE   1

IMPLEMENT_DYNAMIC(CFileExplorerDlg, CBaseDialog)

/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent Window
/////////////////////////////////////////////////////////////////////////////
CFileExplorerDlg::CFileExplorerDlg(CWnd* pParent /*=NULL*/)
	: CBaseDialog(CFileExplorerDlg::IDD, pParent)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CFileExplorerDlg::~CFileExplorerDlg()
{
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV support
///   \param pDX 
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::DoDataExchange(CDataExchange* pDX)
{
  CBaseDialog::DoDataExchange(pDX);
  DDX_Control(pDX, IDC_CB_CHANNELSELECTION, m_cChannelCtrl);
  DDX_Control(pDX, IDC_FILELIST, m_cFileListCtrl);
}


BEGIN_MESSAGE_MAP(CFileExplorerDlg, CBaseDialog)
  ON_BN_CLICKED(IDC_BTN_REFRESH, OnBnClickedBtnRefresh)
  ON_BN_CLICKED(IDC_BTN_UPLOAD, OnBnClickedBtnUpload)
  ON_BN_CLICKED(IDC_BTN_DELETE, OnBnClickedBtnDelete)
  ON_CBN_SELCHANGE(IDC_CB_CHANNELSELECTION, OnCbnSelchangeCbChannelselection)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
/// Thread function (stubbed out, as we don't have a thread)
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::ThreadFunction(void)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Called when the user changes the active device
///   \param pcDevice Device that has been selected, NULL if no device is open
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  m_cChannelCtrl.EnableWindow(FALSE);

  if(m_cChannelCtrl.GetCurSel() == -1)
  {
    m_cChannelCtrl.SetCurSel(0);
  }

  if(NULL != pcDevice)
  {
    if(pcDevice->IsSystemDevice())
    {
      m_cChannelCtrl.EnableWindow(TRUE);
    } else
    {
      unsigned long ulChannel = pcDevice->GetChannelNr();
      
      for(int iIdx = 0; iIdx < m_cChannelCtrl.GetCount(); iIdx++)
      {
        if(m_cChannelCtrl.GetItemData(iIdx) == ulChannel)
        {
          m_cChannelCtrl.SetCurSel(iIdx);
          break;
        }
      }
    }

    UpdateFileList();
  }
}

/////////////////////////////////////////////////////////////////////////////
/// First time dialog initialization
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CFileExplorerDlg::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  /* Add all entries to channel combobox */
  static struct
  {
    TCHAR*        szText;
    unsigned long ulChannelNr;
  } s_atChannels[] =
  {
    {_T("SYSTEM"), 0xFFFFFFFF},
    {_T("PORT_0"), 0},
    {_T("PORT_1"), 1},
    {_T("PORT_2"), 2},
    {_T("PORT_3"), 3},
    {_T("PORT_4"), 4},
    {_T("PORT_5"), 5},
  };

  for(int iChannel = 0; iChannel < sizeof(s_atChannels) / sizeof(s_atChannels[0]); iChannel++)
  {
    int iIdx = m_cChannelCtrl.AddString(s_atChannels[iChannel].szText);
    m_cChannelCtrl.SetItemData(iIdx, s_atChannels[iChannel].ulChannelNr);
  }

  /* Add all columns to file listing control */
  static struct
  {
    TCHAR*        szHeading;
    unsigned char bWidth;
    unsigned long ulAlignment;

  } s_atFileColumns [] =
  {
    {_T("Filename"),   60, LVCFMT_LEFT},
    {_T("Filesize"),   40, LVCFMT_LEFT},
  };

  CRect cRect;
  m_cFileListCtrl.GetWindowRect(cRect);
  cRect.right -= (2 * GetSystemMetrics(SM_CXEDGE) + GetSystemMetrics(SM_CXVSCROLL));

  for(int iCol = 0 ; iCol < sizeof(s_atFileColumns) / sizeof(s_atFileColumns[0]); iCol++)
  {
    m_cFileListCtrl.InsertColumn(iCol, 
                                 s_atFileColumns[iCol].szHeading,
                                 s_atFileColumns[iCol].ulAlignment,
                                 cRect.Width() * s_atFileColumns[iCol].bWidth / 100);
  }

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Updates the file listing, according to current channel selection
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::UpdateFileList()
{
  int iSel = m_cChannelCtrl.GetCurSel();
  ASSERT(-1 != iSel);
  unsigned long ulChannel = (unsigned long)m_cChannelCtrl.GetItemData(iSel);

  m_cFileListCtrl.DeleteAllItems();
  m_cvDirectory.clear();

  CIFX_DIRECTORYENTRY tDirEntry = {0};

  long lRet = CcifXTestDlg::s_pcDevice->FindFirstFile(ulChannel, &tDirEntry);

  if(CIFX_NO_ERROR == lRet)
  {
    AddFile(&tDirEntry);

    while(CIFX_NO_ERROR == (lRet = CcifXTestDlg::s_pcDevice->FindNextFile(ulChannel, &tDirEntry)))
    {
      AddFile(&tDirEntry);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Add a file to the dialog
///   \param ptDirEntry Directory entry
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::AddFile(CIFX_DIRECTORYENTRY* ptDirEntry)
{
  /* Only print entries if it is not a directory */
  if(ptDirEntry->bFiletype != RCX_DIR_LIST_CNF_FILE_TYPE_DIRECTORY)
  {
    m_cvDirectory.push_back(*ptDirEntry);
    unsigned long ulIdx = (unsigned long)(m_cvDirectory.size() - 1);

    CString csTemp(ptDirEntry->szFilename);
    int iItem = m_cFileListCtrl.InsertItem(ulIdx, csTemp);

    csTemp.Format(_T("%u"), ptDirEntry->ulFilesize);
    m_cFileListCtrl.SetItemText(iItem, COLUMN_FILESIZE, csTemp);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Refreshes the file list
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::OnBnClickedBtnRefresh()
{
  UpdateFileList();
}

/////////////////////////////////////////////////////////////////////////////
/// Refreshes the file list
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::OnCbnSelchangeCbChannelselection()
{
  UpdateFileList();
}

void CFileExplorerDlg::OnBnClickedBtnUpload()
{
  int iSelCount = m_cFileListCtrl.GetSelectedCount();

  if(0 == iSelCount)
  {
    AfxMessageBox(_T("Please select a file from the list, to upload first!"));

  } else
  {
    int           iItem     = m_cFileListCtrl.GetSelectionMark();
    unsigned long ulChannel = (unsigned long)m_cChannelCtrl.GetItemData(m_cChannelCtrl.GetCurSel());

    CIFX_DIRECTORYENTRY& tEntry = m_cvDirectory[iItem];

    CString csFileName(tEntry.szFilename);

    CFileDialog cDlg(FALSE, NULL, csFileName, OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT, _T("All Files (*.*)|*.*||"));

    if(IDOK == cDlg.DoModal())
    {
      uint32_t       ulFileSize = tEntry.ulFilesize;
      unsigned char* pbBuffer   = new unsigned char[ulFileSize];

      long lRet = CcifXTestDlg::s_pcDevice->Upload(tEntry.szFilename, ulChannel, pbBuffer, &ulFileSize);

      if(CIFX_NO_ERROR != lRet)
      {
        CString csTemp;
        csTemp.Format(_T("Error uploading file. lRet = 0x%08X!"),
                      lRet);
        AfxMessageBox(csTemp);

      } else
      {
        CFile cFile;
        
        if(!cFile.Open(cDlg.GetPathName(), CFile::modeCreate | CFile::modeWrite))
        {
          AfxMessageBox(_T("Error opening file for writing!"));

        } else
        {
          cFile.Write(pbBuffer, ulFileSize);
        }
      }

      delete [] pbBuffer;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Delete a selected file
/////////////////////////////////////////////////////////////////////////////
void CFileExplorerDlg::OnBnClickedBtnDelete()
{
  int iSelCount = m_cFileListCtrl.GetSelectedCount();

  if(0 == iSelCount)
  {
    AfxMessageBox(_T("Please select a file from the list to delete!"));

  } else
  {
    int           iItem     = m_cFileListCtrl.GetSelectionMark();
    unsigned long ulChannel = (unsigned long)m_cChannelCtrl.GetItemData(m_cChannelCtrl.GetCurSel());

    CIFX_DIRECTORYENTRY&   tEntry     = m_cvDirectory[iItem];
    CIFX_PACKET            tSend      = {0};
    RCX_FILE_DELETE_REQ_T* ptReq      = (RCX_FILE_DELETE_REQ_T*)&tSend;
    RCX_FILE_DELETE_CNF_T  tCnf       = {0};
    unsigned long          ulFileLen  = (unsigned long)strlen(tEntry.szFilename);

    ptReq->tHead.ulDest           = 0x00000020;
    ptReq->tHead.ulSrc            = *(unsigned long*)&(CcifXTestDlg::s_pcDevice);
    ptReq->tHead.ulCmd            = RCX_FILE_DELETE_REQ;
    ptReq->tHead.ulLen            = ulFileLen + sizeof(ptReq->tData) + 1;
    ptReq->tData.ulChannelNo      = ulChannel;
    ptReq->tData.usFileNameLength = (unsigned short)(ulFileLen + 1);

    strcpy((char*)(ptReq + 1), tEntry.szFilename);

    long lRet = CcifXTestDlg::s_pcDevice->PutPacket(&tSend, 1000);

    if( (CIFX_NO_ERROR == lRet) &&
        (CIFX_NO_ERROR == (lRet = CcifXTestDlg::s_pcDevice->GetPacket((CIFX_PACKET*)&tCnf, 
                                                                      sizeof(tCnf), 
                                                                      1000))) &&
        (0 == tCnf.tHead.ulSta) )
    {

      UpdateFileList();

    } else
    {
      CString csError;
      csError.Format(_T("Error deleting file (lRet=0x%08X, ulSta=0x%08X!"),
                     lRet,
                     tCnf.tHead.ulSta);
      AfxMessageBox(csError);
    }
  }

}
